/*
 * Copyright (c) 2001 Personal Media Corporation. All rights reserved.
 */

#ifndef _BTK_H_
#define _BTK_H_
#include "stddef.h"
#include "typedef.h"
#include "btk/btkfwd.h"
#include "btk/btk_color.h"

namespace BToolkit
{

class ColorModel
{
public:
    long pData;
    long pixel_bits;
};

class GC
{
public:
    LATTR attr;
    PAT pat;
    DEG angle;
    DCM mode;
    GRGN* clip;

    GC();
    void setLineWidth(int w);
    void setLineType(LineKind t);
    void setForegroundColor(COLOR fg);
    void setBackgroundColor(COLOR bg);
    COLOR getForegroundColor() const { return pat.spat.fgcol; }
    COLOR getBackgroundColor() const { return pat.spat.bgcol; }

private:
    GC(const GC&);
    GC&operator=(const GC&);
};

class Drawable
{
public:
    virtual ~Drawable() = 0;
    virtual void getDimensions(int& aWidth, int& aHeight) = 0;

    virtual void drawLine(const GC&, int x0, int y0, int x1, int y1) = 0;
    virtual void drawRect(const GC&, int x, int y, int w, int h, bool fill = false) = 0;
    virtual void drawRoundRect(const GC&, int x, int y, int w, int h, int rw, int rh, bool fill = false) = 0;
    virtual void drawOval(const GC&, int x, int y, int w, int h, bool fill = false) = 0;
    virtual void drawArc(const GC&, int x, int y, int w, int h, int sang, int aang, bool fill = false) = 0;
    virtual void drawPolyLine(const GC&, const POLY& pp) = 0;
    virtual void drawPolygon(const GC&, const POLY& pp, bool fill = false) = 0;
    virtual void drawRect3D(const GC&, int x, int y, int w, int h, int thickness, bool raised = true, bool fill = false) = 0;
    virtual void drawUnicodeString(const GC&, const FontMetrics&, int& x, int& y, const UH*, unsigned int length) = 0;

    virtual bool copyArea(const GC& gc, int x, int y,
                          Drawable&, int dX, int dY, int dW, int dH) = 0;
    virtual bool copyBMP(const GC& gc, int dx, int dy,
                         const BMP&, const unsigned char* mask,
                         int sx, int sy, int w, int h) = 0;
    virtual bool copyBmpScale(const GC& gc, int dx, int dy, int dw, int dh,
                              const BMP& bmp, const unsigned char* mask,
                              int sx, int sy, int w, int h) = 0;
    virtual bool copyBmpTile (const GC& gc, int x0, int y0, int x1, int y1,
                              const BMP& bmp, const unsigned char* mask,
                              int xoff, int yoff) = 0;
    virtual Pixmap* getImage(int x, int y, int w, int h) = 0;

    virtual GID startDrawing() = 0;
    virtual void endDrawing() = 0;
    virtual void applyDrawingClip(GRGN*) = 0;
};

class Pixmap : public Drawable
{
public:
    virtual ~Pixmap() = 0;

    virtual const BMP& getBMP() const = 0;
    virtual unsigned char* getBits() = 0;
    virtual int getDepth() const = 0;
    virtual int getStride() const = 0;
    virtual int getRowBytes() const = 0;
    virtual void dispose() = 0;
};


class EventHandler
{
public:
    virtual ~EventHandler() = 0;

    // default callback
    virtual void handleActivate();
    virtual void handleDeactivate();
    virtual void handleExposed(int x, int y, int w, int h);
    virtual void handleMoved(int x, int y);
    virtual void handleResized(int width, int height);
    virtual void handleMouseEvent(long id, long when, long modifiers,
                                  long x , long y, long clickcount, bool popuptrigger);
    virtual void handleKeyEvent(long id, long when, long modifiers, long keycode, long keychar);
    virtual void handleFocusEvent(long id, bool isTemp);
    virtual void handleQuit();
    virtual void handleDestroy();
};


class Component : public Drawable
{
public:
    virtual ~Component() = 0;

    virtual void dispose() = 0;
    virtual Component* getParent() = 0;
    virtual Component* getRootComp() = 0;
    virtual void show() = 0;
    virtual void hide() = 0;
    virtual bool isVisible() = 0;
    virtual void move(int x, int y) = 0;
    virtual void resize(int w, int h) = 0;
    virtual void moveResize(int x, int y, int w, int h) = 0;
    virtual void scroll(int x, int y, const RECT* clip) = 0;
    virtual bool isEnable () = 0;
    virtual void setEnable(bool) = 0;
    virtual void setFocus() = 0;
    virtual void getBounds(RECT&) = 0;
    virtual void getGlobalOrigin(int& x, int& y) = 0;
    virtual COLOR getForegroundColor() = 0;
    virtual void setForegroundColor(const COLOR&) = 0;
    virtual COLOR getBackgroundColor() = 0;
    virtual void setBackgroundColor(const COLOR&) = 0;
    virtual void setFont(const Font&) = 0;
    virtual Font* getFont() = 0;
    virtual void setCursor(const PTRSTL&) = 0;
    virtual PTRSTL getCursor() = 0;
    virtual void toFront() = 0;
    virtual void toBack() = 0;
    virtual void grabPointer() = 0;
    virtual void ungrabPointer() = 0;

    virtual void setTitleWithUnicode(const UH*, size_t aLength) = 0;

    virtual void installEventHandler(EventHandler*) = 0;
    virtual void removeEventHandler(EventHandler*) = 0;


    virtual int getNativeWindowId () = 0;
    virtual void* pData() = 0; // get implementation private data
};

class Insets
{
public:
    virtual ~Insets() = 0;

    virtual void getInsets(long& top, long& left, long& right, long& bottom) = 0;
    virtual void setInsets(long top, long left, long right, long bottom) = 0;
};

class Frame : public Component, public Insets
{
public:
    virtual ~Frame() = 0;

    virtual void reshape(long x, long y, long w, long h) = 0;
    virtual void setResizable(bool resizable) = 0;

    virtual void setInputContext (BtkIc*) = 0;
};

class FontMetrics
{
public:
    virtual ~FontMetrics() = 0;

    virtual void calcUnicodeStringBounds(const UH* aString, int aLength, RECT& aBounds) = 0;
    virtual const FSSPEC& fsspec() const = 0;
    virtual const FNTINFO& fntinfo() const = 0;
    virtual int ascent() const = 0;
    virtual int descent() const = 0;
    virtual int leading() const = 0;
    virtual int height() const = 0;
    virtual int advance() const = 0;
    virtual int spaceWidth() const = 0;
};

class ComponentFactory
{
public:
    virtual ~ComponentFactory() = 0;

    virtual Pixmap* makePixmap(int aWidth, int aHeight) = 0;
    virtual Component* makeComponent(const RECT& aBounds, Component* aParent) = 0;
    virtual Frame* makeFrame(const RECT& aBounds, Component* aParent) = 0;
    virtual Frame* makePopup(const RECT& aBounds, Component* aParent) = 0;
    virtual Frame* makeDialog(const RECT& aBounds, Component* aParent) = 0;
    virtual FontMetrics* makeFontMetrics(const TC* aFamily, unsigned long aClass, int aAttr, int aHSize, int aVSize) = 0;
};

class Toolkit : public ComponentFactory
{
public:
    virtual ~Toolkit() = 0;

    virtual void run() = 0;
    virtual void exit() = 0;
    virtual long processEvent() = 0;
    virtual void registerWaitingCallback(void(*)(void*), void* arg) = 0;
    virtual void registerMessageReceivedCallback (bool (*) (void)) = 0;

    virtual long getScrollbarWidth() = 0;
    virtual void getScreenWorkArea(RECT& r) = 0;
    virtual long getScreenWidth() const = 0;
    virtual long getScreenHeight() const = 0;
    virtual long getScreenResolution() const = 0;
    virtual ColorModel* makeColorModel() = 0;

    virtual BtCompWin* getScreenWindow() const = 0;
    virtual GID getScreenGID() const = 0;
    virtual const TC* getScreenDevice() const = 0;
    virtual const DEV_SPEC& getScreenDevSpec() const = 0;
    virtual const CSPEC& getScreenCSpec() const = 0;
    virtual size_t getScreenCSpecColormapSize() const = 0;

    virtual GID makeScratchGID() = 0;
    virtual void disposeScratchGID(GID) = 0;

    virtual void constrainPosition(int& x, int& y) const = 0;

    virtual bool queryNeedUpdate() const = 0;
    virtual void setNeedUpdate(bool value) = 0;

    static Toolkit* defaultToolkit();
};

namespace event {

class InputEvent
{
public:
    static const long COMPONENT_FIRST   = 100L;
    static const long COMPONENT_MOVED   = 100L;
    static const long COMPONENT_RESIZED = 101L;
    static const long COMPONENT_SHOWN   = 102L;
    static const long COMPONENT_HIDDEN  = 103L;
    static const long COMPONENT_LAST    = 103L;

    static const long SHIFT_MASK = 1L;
    static const long CTRL_MASK = 2L;
    static const long META_MASK = 4L;
    static const long ALT_MASK = 8L;
    static const long BUTTON1_MASK = 16L;
    static const long BUTTON2_MASK = 8L;
};

class MouseEvent
{
public:
    static const long MOUSE_FIRST    = 500L;
    static const long MOUSE_CLICKED  = 500L;
    static const long MOUSE_PRESSED  = 501L;
    static const long MOUSE_RELEASED = 502L;
    static const long MOUSE_MOVED    = 503L;
    static const long MOUSE_ENTERED  = 504L;
    static const long MOUSE_EXITED   = 505L;
    static const long MOUSE_DRAGGED  = 506L;
    static const long MOUSE_LAST     = 506L;
};

class KeyEvent
{
public:
    static const long KEY_FIRST    = 400L;
    static const long KEY_TYPED    = 400L;
    static const long KEY_PRESSED  = 401L;
    static const long KEY_RELEASED = 402L;
    static const long KEY_LAST     = 402L;
};

class FocusEvent
{
public:
    static const long FOCUS_FIRST  = 1004L;
    static const long FOCUS_GAINED = 1004L;
    static const long FOCUS_LOST   = 1005L;
    static const long FOCUS_LAST   = 1005L;
};

} // event
using namespace event;

} // namespace BToolkit

#endif
// Local Variables:
// mode: C++
// c-file-style: "stroustrup"
// End:
